"use client";

import Image from "next/image";
import Link from "next/link";
import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselDots, // Ensure this is exported from your carousel component file
} from "@/components/ui/carousel";
import { useRef } from "react";
import Autoplay from "embla-carousel-autoplay";
import { ArrowRight } from "lucide-react";

export default function ScrollTop({ data }) {
  // Slow down the autoplay for a more elegant feel
  const plugin = useRef(Autoplay({ delay: 4000, stopOnInteraction: true }));

  const activeCategories = data?.items?.filter(
    (cat) => cat.category_status === 1 && cat.category_image !== null,
  );

  if (!activeCategories?.length) return null;

  return (
    <section className="w-full py-16 px-4 md:px-8 bg-gradient-to-b from-white to-stone-50/50">
      <div className="max-w-[1600px] mx-auto">
        {/* Header Section with Decorative Elements */}
        <div className="text-center mb-12 space-y-4">
          <span className="uppercase tracking-[0.2em] text-xs font-medium text-brand-gold/80">
            Discover Our Collection
          </span>
          <h2 className="text-4xl md:text-5xl font-cormorant text-primary font-light">
            {data?.name || "Shop by Category"}
          </h2>
          <div className="w-24 h-[1px] bg-brand-gold/40 mx-auto mt-4" />
        </div>

        {/* Carousel Wrapper */}
        <div className="relative group/carousel">
          <Carousel
            plugins={[plugin.current]}
            onMouseEnter={plugin.current.stop}
            onMouseLeave={plugin.current.play}
            opts={{
              align: "center",
              loop: true,
              slidesToScroll: 1,
              dragFree: true,
            }}
            className="w-full"
          >
            <CarouselContent className="-ml-4 md:-ml-8 py-4">
              {activeCategories.map((category) => {
                const img = category.category_image?.replace("/api", "");
                return (
                  <CarouselItem
                    key={category.id}
                    // Adjusted responsive widths for a spacious look
                    className="pl-4 md:pl-8 basis-3/4 sm:basis-1/2 md:basis-1/3 lg:basis-1/4 xl:basis-1/5"
                  >
                    <Link
                      href={`/${category.category_url}`}
                      className="block group/card h-full"
                    >
                      {/* Card Container */}
                      <div className="relative h-full flex flex-col items-center">
                        {/* Image Frame - Arch Shape */}
                        <div className="relative w-full aspect-[3/4] overflow-hidden rounded-t-[100px] rounded-b-xl border border-stone-200 bg-white shadow-sm transition-all duration-500 ease-out group-hover/card:shadow-xl group-hover/card:border-brand-gold/50">
                          {/* Inner Border (The "Jewelry Box" effect) */}
                          <div className="absolute inset-3 rounded-t-[90px] rounded-b-lg border border-white/20 z-10 pointer-events-none group-hover/card:border-white/40 transition-colors" />

                          <Image
                            src={img}
                            alt={category.name}
                            fill
                            className="object-cover transition-transform duration-700 ease-in-out group-hover/card:scale-110"
                            sizes="(max-width: 640px) 75vw, (max-width: 1024px) 33vw, 20vw"
                          />

                          {/* Subtle Overlay on Hover */}
                          <div className="absolute inset-0 bg-primary/0 transition-colors duration-500 group-hover/card:bg-primary/10" />
                        </div>

                        {/* Text Content */}
                        <div className="pt-6 pb-2 text-center w-full relative">
                          {/* Animated decorative line */}
                          <div className="absolute top-0 left-1/2 -translate-x-1/2 w-0 h-[1px] bg-brand-gold transition-all duration-500 group-hover/card:w-12" />

                          <h3 className="text-2xl font-cormorant font-bold text-primary group-hover/card:text-brand-gold transition-colors duration-300">
                            {category.name}
                          </h3>

                          <div className="flex items-center justify-center gap-2 mt-2 opacity-0 transform translate-y-2 transition-all duration-300 group-hover/card:opacity-100 group-hover/card:translate-y-0">
                            <span className="text-xs uppercase tracking-widest font-lato text-gray-500">
                              View
                            </span>
                            <ArrowRight className="w-3 h-3 text-brand-gold" />
                          </div>
                        </div>
                      </div>
                    </Link>
                  </CarouselItem>
                );
              })}
            </CarouselContent>

            {/* Dot Navigation */}
            <div className="mt-8 flex justify-center w-full">
              <CarouselDots
                activeColor="bg-brand-gold"
                inactiveColor="bg-primary/20"
              />
            </div>
          </Carousel>
        </div>
      </div>
    </section>
  );
}
